package jcircus.environment;

import java.util.Hashtable;
import java.util.Iterator;
import jcircus.exceptions.ChanUseUnificationException;
import jcircus.exceptions.DifferentCardinalitiesException;
import jcircus.exceptions.MoreThanOneWriterException;
import jcircus.exceptions.NotYetImplementedException;
import jcircus.util.ChanUse;

/**
 * ProcChanUseEnv.java
 *
 * @author Angela Freitas
 *
 */
public class ProcChanUseEnv {
    
    /*
     * Integer -> ChanUse
     */
    private Hashtable hashtable_;
    
    /*
     *
     */
    private int cardinality_;
    
    /**
     *
     */
    public ProcChanUseEnv(int cardinality) {
        this.hashtable_ = new Hashtable();
        this.cardinality_ = cardinality;
    }
    
    public int getCardinality() {
        return this.cardinality_;
    }
    
    
    /**
     *
     */
    public void put(Integer procId, ChanUse chanUse) {
        
        this.hashtable_.put(procId, chanUse);
    }
    
    public ChanUse get(Integer procId) {
        return (ChanUse) this.hashtable_.get(procId);
    }
    
    public void putAll(ProcChanUseEnv other) {
        
        Iterator iterator = other.iteratorKeys();
        ChanUse javaTypeChannel;
        
        while(iterator.hasNext()) {
            Integer procId = (Integer) iterator.next();
            javaTypeChannel = other.get(procId);
            
            this.put(procId, javaTypeChannel);
        }
        
    }
    
    public ProcChanUseEnv merge(ProcChanUseEnv other, boolean isParallel) 
            throws MoreThanOneWriterException, ChanUseUnificationException,
                DifferentCardinalitiesException {
        
        int c;
        
        // Checks if there is more than one writer in the parallelism
        if (isParallel && this.containsWriter() && other.containsWriter())
            throw new MoreThanOneWriterException();
        
        // Checks if unification is possible
        if (!isParallel && 
                (this.containsWriter() && other.containsReader() || 
                 this.containsReader() && other.containsWriter()))
                    
            throw new ChanUseUnificationException();

        
        // resolves the cardinality
        if (isParallel)
            c = this.cardinality_ + other.cardinality_;
        else {
            
            /**
             * If this is not a parallelism, then the cardinalities must be the
             * same, otherwise we will have complicated cases to deal with, ex.:
             *
             * ((a -> SKIP [| a |] a -> SKIP) |~| (a -> SKIP))
             * [| a |]
             * (a -> SKIP)
             *
             * In this case we have diferent cardinalities in each of the parts 
             * of the internal choice. Depending on which side is chosen, we would
             * have a single-synchronization or a multi-synchronization, and this
             * would only be determined at runtime.
             *
             * ((a -> SKIP [| a |] a -> SKIP); (a -> SKIP))
             * [| a |]
             * (a -> SKIP); (a -> SKIP)
             *
             * In this case we have diferent cardinalities in each of the parts 
             * of the seq. composition. This leads to a situation where the first
             * synchronization on 'a' is a multi-synchronization, and the second 
             * is a single one.
             *
             */
            if (this.cardinality_ != other.cardinality_) {
                throw new DifferentCardinalitiesException();
            } 
            
            // The cardinality of the compound process is the same as 
            // each of the composite ones
            c = this.cardinality_;
            
        }
        
        ProcChanUseEnv result = new ProcChanUseEnv(c);
        
        if (isParallel) {
            
            // Leave as it is.
            // I don't expect a ProcessAlreadyDefinedInSyncEnvException to be 
            // thrown in any of the next two putAll - There will be not the 
            // same keys in 'this' and 'other' environments, because we do not 
            // consider repetition of process in a parallelism
            result.putAll(this);
            result.putAll(other);
            
        } else {
            // If this is not a parallelism, the use of all of them has to be
            // unified
            ChanUse j;
            
            if (this.containsReader() || other.containsReader()) {
                j = ChanUse.Input;
            } else if (this.containsWriter() || other.containsWriter()) {
                j = ChanUse.Output;
            } else {
                j = ChanUse.Undefined;
            }
            
            Iterator itThis = this.iteratorKeys();
            Iterator itOther = other.iteratorKeys();

            // Check ProcessAlreadyDefinedInSyncEnvException here ?????
            while (itThis.hasNext()) {               
                Integer procId = (Integer) itThis.next();
                result.put(procId, j);
            }
            
            while (itOther.hasNext()) {
                Integer procId = (Integer) itOther.next();
                result.put(procId, j);
            }
        }
        
        return result;
    }
    
    private boolean containsWriter() {
        return this.hashtable_.contains(ChanUse.Output);
    }
    
    private boolean containsReader() {
        return (this.hashtable_.contains(ChanUse.Input) || this.hashtable_.contains(ChanUse.AltInput));
    }
    
    public Iterator iteratorKeys() {
        return this.hashtable_.keySet().iterator();
    }
    
    private boolean containsKey(Integer procId) {
        return this.hashtable_.containsKey(procId);
    }
    
    /**
     * Multi-synchronization.
     */
    public boolean isMultiSync() {
        return (this.cardinality_ > 2);
    }
    
    /**
     * Single-synchronization or multi-synchronization.
     */
    public boolean isSync() {
        return (this.cardinality_ > 1);
    }
    
    /**
     *
     */
    
    public ChanUse getChanUseGuiNotSyncChannel() {
        
        ChanUse javaTypeChannel;
        
        if (this.isSync())
            //throw new Exception("This is a synchronized channel");
            // I am calling this method only when I know that this channel
            // is not synchronized, that's why I decided to return null 
            // instead of throwing an exception
            return null;
        
        
        if (this.containsReader() || (!this.containsReader() && !this.containsWriter())) {
            // Se tiver um reader ou todos forem undefined
            javaTypeChannel = ChanUse.Input;
        } else {
            // Se tiver um writer 
            javaTypeChannel = ChanUse.Output;
        }
        //(Obs: nao pode conter um reader e um writer porque ele nao  Sync)
        
        return javaTypeChannel;
    }
    
    /**
     * This method must only be called for basic process.
     *
     * Replaces the id in this environment for the new id
     * passed as parameter.
     */
    public ProcChanUseEnv replaceId(Integer newId, Integer oldId) {
        
        ChanUse chanUse = null;
        
        // Creates a new object
        ProcChanUseEnv r = new ProcChanUseEnv(1);
        
        // Has to have size one because it is a basic process
        if (hashtable_.size() != 1)
            throw new RuntimeException("Method replaceIds has been called for a non-basic process.");
            
        Iterator it = iteratorKeys();

        // Iterates this array - only one element
        while(it.hasNext()) {
            // Gets the old id
            oldId = (Integer) it.next();
            chanUse = get(oldId);
        }

        // inserts the new id
        r.put(newId, chanUse);
        
        // Returns the new object
        return r;
    }
    
    public int size() {
        return this.hashtable_.size();                
    }
    
    public String print(int nTabs) {
        
        String tabs = "";
        while(nTabs > 0) {
            tabs = tabs + "\t";
            nTabs--;
        }
            
        StringBuilder s = new StringBuilder();
        s.append(tabs + "ChanMSEnv, cardinality = " + this.cardinality_ + "\n");
        Iterator it = this.iteratorKeys();
        
        while(it.hasNext()) {
            Integer procId = (Integer) it.next();
            ChanUse jtc = this.get(procId);
            s.append(tabs + procId + " -> " + jtc.toStringGUI() + "\n");
        }
        return s.toString();
    }
}
